/* ==========================================================
   PDF Pro — Translator Engine
   Priority fallback: External APIs → Offline Dictionary
   API details are hidden from the UI.
   ========================================================== */

window.Translator = (function () {
  'use strict';

  /* ── Hindi Unicode range & basic dictionary ── */
  const HINDI_DICT = {
    // Greetings & common
    'hello': 'नमस्ते', 'hi': 'नमस्ते', 'goodbye': 'अलविदा', 'yes': 'हाँ', 'no': 'नहीं',
    'please': 'कृपया', 'thank you': 'धन्यवाद', 'thanks': 'शुक्रिया', 'sorry': 'माफ़ करें',
    'welcome': 'स्वागत है', 'good morning': 'सुप्रभात', 'good night': 'शुभ रात्रि',
    'how are you': 'आप कैसे हैं', 'i am fine': 'मैं ठीक हूँ', 'what': 'क्या', 'where': 'कहाँ',
    'when': 'कब', 'who': 'कौन', 'why': 'क्यों', 'how': 'कैसे',
    // Common words
    'the': '', 'a': 'एक', 'is': 'है', 'are': 'हैं', 'was': 'था', 'were': 'थे',
    'this': 'यह', 'that': 'वह', 'these': 'ये', 'those': 'वे', 'and': 'और',
    'or': 'या', 'but': 'लेकिन', 'if': 'अगर', 'then': 'तब', 'because': 'क्योंकि',
    'so': 'इसलिए', 'also': 'भी', 'not': 'नहीं', 'very': 'बहुत', 'good': 'अच्छा',
    'bad': 'बुरा', 'big': 'बड़ा', 'small': 'छोटा', 'new': 'नया', 'old': 'पुराना',
    'first': 'पहला', 'last': 'अंतिम', 'many': 'कई', 'some': 'कुछ', 'all': 'सभी',
    'time': 'समय', 'day': 'दिन', 'night': 'रात', 'year': 'साल', 'month': 'महीना',
    'today': 'आज', 'tomorrow': 'कल', 'yesterday': 'कल', 'now': 'अभी', 'then': 'फिर',
    'people': 'लोग', 'person': 'व्यक्ति', 'man': 'आदमी', 'woman': 'महिला', 'child': 'बच्चा',
    'home': 'घर', 'work': 'काम', 'school': 'स्कूल', 'city': 'शहर', 'country': 'देश',
    'water': 'पानी', 'food': 'खाना', 'money': 'पैसा', 'life': 'जीवन', 'love': 'प्यार',
    'name': 'नाम', 'number': 'नंबर', 'information': 'जानकारी', 'help': 'मदद',
    'document': 'दस्तावेज़', 'file': 'फ़ाइल', 'page': 'पृष्ठ', 'text': 'पाठ',
    'translate': 'अनुवाद', 'language': 'भाषा', 'english': 'अंग्रेज़ी', 'hindi': 'हिंदी',
    'india': 'भारत', 'world': 'दुनिया', 'important': 'महत्वपूर्ण', 'need': 'ज़रूरत',
    'want': 'चाहना', 'can': 'कर सकता', 'will': 'करेगा', 'has': 'के पास है', 'have': 'है',
    'see': 'देखना', 'know': 'जानना', 'think': 'सोचना', 'say': 'कहना', 'go': 'जाना',
    'come': 'आना', 'make': 'बनाना', 'give': 'देना', 'take': 'लेना', 'use': 'उपयोग',
    'read': 'पढ़ना', 'write': 'लिखना', 'send': 'भेजना', 'receive': 'प्राप्त',
  };

  const ENGLISH_DICT = Object.fromEntries(Object.entries(HINDI_DICT).map(([k,v]) => [v,k]));

  /* ── Chunk text ── */
  function chunkText(text, size = 200) {
    const sentences = text.match(/[^.!?]+[.!?\n]*/g) || [text];
    const chunks = [];
    let current  = '';
    for (const s of sentences) {
      if ((current + s).length > size && current) {
        chunks.push(current.trim());
        current = s;
      } else {
        current += s;
      }
    }
    if (current.trim()) chunks.push(current.trim());
    return chunks.length ? chunks : [text];
  }

  /* ── MyMemory API ── */
  async function tryMyMemory(text, srcLang, tgtLang) {
    const url = `https://api.mymemory.translated.net/get?q=${encodeURIComponent(text)}&langpair=${srcLang}|${tgtLang}`;
    const r = await fetch(url, { signal: AbortSignal.timeout(8000) });
    if (!r.ok) throw new Error('MyMemory: ' + r.status);
    const data = await r.json();
    if (data.responseStatus !== 200) throw new Error('MyMemory error');
    return data.responseData.translatedText;
  }

  /* ── Lingva API ── */
  async function tryLingva(text, srcLang, tgtLang) {
    const url = `https://lingva.ml/api/v1/${srcLang}/${tgtLang}/${encodeURIComponent(text)}`;
    const r = await fetch(url, { signal: AbortSignal.timeout(8000) });
    if (!r.ok) throw new Error('Lingva: ' + r.status);
    const data = await r.json();
    if (!data.translation) throw new Error('Lingva: no translation');
    return data.translation;
  }

  /* ── LibreTranslate ── */
  async function tryLibre(text, srcLang, tgtLang, endpoint) {
    const r = await fetch(endpoint || 'https://libretranslate.com/translate', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ q: text, source: srcLang, target: tgtLang, format: 'text' }),
      signal: AbortSignal.timeout(10000),
    });
    if (!r.ok) throw new Error('LibreTranslate: ' + r.status);
    const data = await r.json();
    if (!data.translatedText) throw new Error('LibreTranslate: empty');
    return data.translatedText;
  }

  /* ── Argos ── */
  async function tryArgos(text, srcLang, tgtLang) {
    return tryLibre(text, srcLang, tgtLang, 'https://translate.argosopentech.com/translate');
  }

  /* ── Google (if key provided) ── */
  async function tryGoogle(text, srcLang, tgtLang, key) {
    const url = `https://translation.googleapis.com/language/translate/v2?key=${key}`;
    const r = await fetch(url, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ q: text, source: srcLang, target: tgtLang }),
      signal: AbortSignal.timeout(8000),
    });
    if (!r.ok) throw new Error('Google: ' + r.status);
    const data = await r.json();
    return data.data.translations[0].translatedText;
  }

  /* ── DeepL (if key provided) ── */
  async function tryDeepL(text, srcLang, tgtLang, key) {
    const r = await fetch('https://api-free.deepl.com/v2/translate', {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded', 'Authorization': 'DeepL-Auth-Key ' + key },
      body: new URLSearchParams({ text, source_lang: srcLang.toUpperCase(), target_lang: tgtLang.toUpperCase() }),
      signal: AbortSignal.timeout(8000),
    });
    if (!r.ok) throw new Error('DeepL: ' + r.status);
    const data = await r.json();
    return data.translations[0].text;
  }

  /* ── Offline Dictionary ── */
  function offlineTranslate(text, direction) {
    const dict   = direction === 'en-hi' ? HINDI_DICT : ENGLISH_DICT;
    const words  = text.toLowerCase().split(/(\s+|[,;:!?.]+)/);
    const result = words.map(token => {
      const clean = token.trim().toLowerCase();
      return dict[clean] !== undefined ? (dict[clean] || '') : token;
    });
    return result.join('');
  }

  /* ── Master translate function ── */
  async function translateChunk(text, direction, onProgress) {
    const [srcLang, tgtLang] = direction === 'en-hi' ? ['en', 'hi'] : ['hi', 'en'];

    const cfg   = window.CONFIG || {};
    const apis  = (cfg.translationAPIs || []).filter(a => a.enabled).sort((a, b) => a.priority - b.priority);

    /* Try enabled configured APIs */
    for (const api of apis) {
      try {
        let result;
        switch (api.id) {
          case 'google':     result = await tryGoogle(text, srcLang, tgtLang, api.key); break;
          case 'deepl':      result = await tryDeepL(text, srcLang, tgtLang, api.key); break;
          case 'mymemory':   result = await tryMyMemory(text, srcLang, tgtLang); break;
          case 'lingva':     result = await tryLingva(text, srcLang, tgtLang); break;
          case 'libre':      result = await tryLibre(text, srcLang, tgtLang, api.endpoint); break;
          case 'argos':      result = await tryArgos(text, srcLang, tgtLang); break;
          default: continue;
        }
        if (result && result.trim()) return { text: result, method: api.name };
      } catch (err) {
        console.debug(`[Translator] ${api.name} failed:`, err.message);
      }
    }

    /* Offline fallback */
    const offline = offlineTranslate(text, direction);
    return { text: offline, method: 'Offline (Dictionary)' };
  }

  /* ── Translate full text (chunked) ── */
  async function translateText(fullText, direction, onProgress) {
    const cfg  = window.CONFIG || {};
    const size = cfg.offlineFallback?.chunkSize || 200;
    const chunks  = chunkText(fullText, size);
    const results = [];
    let   method  = null;

    for (let i = 0; i < chunks.length; i++) {
      const res = await translateChunk(chunks[i], direction, onProgress);
      results.push(res.text);
      if (!method) method = res.method;
      if (onProgress) onProgress(Math.round((i + 1) / chunks.length * 100));
    }

    return { translatedText: results.join('\n'), method };
  }

  /* ── Generate translated PDF ── */
  async function buildTranslatedPDF(translatedText, originalName, direction) {
    const script = document.createElement('script');
    script.src = 'https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js';
    if (!window.jspdf && typeof jsPDF === 'undefined') {
      await new Promise(res => { script.onload = res; document.head.appendChild(script); });
    }

    const jsPDFLib = window.jspdf ? window.jspdf.jsPDF : jsPDF;
    const doc = new jsPDFLib({
      orientation: 'portrait',
      unit: 'mm',
      format: 'a4',
      compress: true,
    });

    // Noto Sans Devanagari subset for Hindi (base64 embedded)
    // For full Unicode support we use the standard font and note the limitation
    doc.setFont('helvetica');
    doc.setFontSize(10);

    const pageW  = doc.internal.pageSize.getWidth();
    const pageH  = doc.internal.pageSize.getHeight();
    const margin = 15;
    const lineH  = 6;
    const maxW   = pageW - margin * 2;

    // Header
    doc.setFontSize(14);
    doc.setTextColor(19, 252, 3);
    doc.text(`Translated Document — ${direction === 'en-hi' ? 'English → Hindi' : 'Hindi → English'}`, margin, 20);
    doc.setFontSize(9);
    doc.setTextColor(150, 150, 150);
    doc.text(`Source: ${originalName} | Generated by PDF Pro`, margin, 27);
    doc.setDrawColor(19, 252, 3);
    doc.line(margin, 30, pageW - margin, 30);

    doc.setFontSize(10);
    doc.setTextColor(30, 30, 30);

    let y = 38;
    const lines = doc.splitTextToSize(translatedText, maxW);

    for (const line of lines) {
      if (y > pageH - margin) {
        doc.addPage();
        y = margin;
      }
      doc.text(line, margin, y);
      y += lineH;
    }

    return new Blob([doc.output('arraybuffer')], { type: 'application/pdf' });
  }

  return { translateText, buildTranslatedPDF, chunkText };
})();
