/* ============================================================
   PDF Pro — PDF Tools Engine  v5.0  (FINAL / PRODUCTION)
   ============================================================

   All 16 tool functions. Every function:
     • validates inputs upfront and rejects with a clear message
     • validates output bytes before returning (where applicable)
     • never resolves with an empty or invalid Blob
     • returns a native Blob that PDFProDownload.trigger() can
       consume directly — no further pre-processing needed

   Public API
   ----------
   PDFTools.compressPDF(file, quality)         → Promise<Blob>
   PDFTools.mergePDFs(files)                   → Promise<Blob>
   PDFTools.splitPDF(file, type, ranges)       → Promise<[{blob,name}]>
   PDFTools.rotatePDF(file, deg, pages)        → Promise<Blob>
   PDFTools.removePages(file, pageList)        → Promise<Blob>
   PDFTools.extractPages(file, pageList)       → Promise<Blob>
   PDFTools.reorderPages(file, newOrder)       → Promise<Blob>
   PDFTools.pdfToJPG(file, dpi, quality)       → Promise<[{blob,name,dataURL}]>
   PDFTools.jpgToPDF(files)                    → Promise<Blob>
   PDFTools.addWatermark(file, text, opts)     → Promise<Blob>
   PDFTools.lockPDF(file, userPwd, ownerPwd)   → Promise<Blob>
   PDFTools.unlockPDF(file, password)          → Promise<Blob>
   PDFTools.pdfToWord(file, progressCb)        → Promise<Blob (.doc)>
   PDFTools.wordToPDF(file, progressCb)        → Promise<Blob (.pdf)>
   PDFTools.getPageCount(file)                 → Promise<number>
   PDFTools.extractText(file, progressCb)      → Promise<string>
   ============================================================ */

window.PDFTools = (function () {
  'use strict';

  /* ── CDN library registry ──────────────────────────────── */
  var CDNS = {
    pdfLib:  'https://cdnjs.cloudflare.com/ajax/libs/pdf-lib/1.17.1/pdf-lib.min.js',
    pdfjs:   'https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf.min.js',
    jspdf:   'https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js',
    mammoth: 'https://cdnjs.cloudflare.com/ajax/libs/mammoth/1.6.0/mammoth.browser.min.js',
  };

  /* Fallback CDNs */
  var CDNS_FB = {
    pdfLib:  'https://unpkg.com/pdf-lib@1.17.1/dist/pdf-lib.min.js',
    pdfjs:   'https://unpkg.com/pdfjs-dist@3.11.174/build/pdf.min.js',
    jspdf:   'https://unpkg.com/jspdf@2.5.1/dist/jspdf.umd.min.js',
    mammoth: 'https://unpkg.com/mammoth@1.6.0/mammoth.browser.min.js',
  };

  var READY = {
    pdfLib:  function () { return typeof PDFLib !== 'undefined' && !!PDFLib.PDFDocument; },
    pdfjs:   function () { return typeof pdfjsLib !== 'undefined'; },
    jspdf:   function () { return !!(window.jspdf && window.jspdf.jsPDF) || typeof jsPDF !== 'undefined'; },
    mammoth: function () { return typeof mammoth !== 'undefined'; },
  };

  var WORKER_URL = 'https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf.worker.min.js';
  var _loadCache = {};

  /* ── Script loader with fallback ──────────────────────── */
  function _loadScript(url, fbUrl) {
    if (_loadCache[url]) return _loadCache[url];
    _loadCache[url] = new Promise(function (resolve, reject) {
      var s = document.createElement('script');
      s.src = url; s.async = true;
      s.onload  = resolve;
      s.onerror = function () {
        if (!fbUrl) { reject(new Error('Library unavailable: ' + url)); return; }
        var s2 = document.createElement('script');
        s2.src = fbUrl; s2.async = true;
        s2.onload  = resolve;
        s2.onerror = function () {
          reject(new Error('Both CDNs failed. Check your internet connection.'));
        };
        document.head.appendChild(s2);
      };
      document.head.appendChild(s);
    });
    return _loadCache[url];
  }

  /* ── Wait for a library global to appear ───────────────── */
  function _waitReady(key, timeoutMs) {
    timeoutMs = timeoutMs || 18000;
    return new Promise(function (resolve, reject) {
      if (READY[key]()) { resolve(); return; }
      var elapsed = 0;
      var tick = setInterval(function () {
        elapsed += 100;
        if (READY[key]())           { clearInterval(tick); resolve(); }
        else if (elapsed >= timeoutMs) { clearInterval(tick); reject(new Error(key + ' timed out loading.')); }
      }, 100);
    });
  }

  function _ensureLib(key) {
    if (READY[key]()) return Promise.resolve();
    return _loadScript(CDNS[key], CDNS_FB[key]).then(function () {
      return _waitReady(key);
    });
  }

  /* ── Library getters ───────────────────────────────────── */
  function _getPDFLib() {
    return _ensureLib('pdfLib').then(function () {
      return window.PDFLib || PDFLib;
    });
  }
  function _getPdfjs() {
    return _ensureLib('pdfjs').then(function () {
      pdfjsLib.GlobalWorkerOptions.workerSrc = WORKER_URL;
      return pdfjsLib;
    });
  }
  function _getJsPDF() {
    return _ensureLib('jspdf').then(function () {
      return (window.jspdf && window.jspdf.jsPDF) || jsPDF;
    });
  }

  /* ── PDF output validation ──────────────────────────────── */
  /* Throws if bytes are empty or missing %PDF- header.       */
  function _assertValidPDF(bytes, ctx) {
    ctx = ctx || 'PDF output';
    if (!bytes || bytes.length === 0)
      throw new Error(ctx + ' produced empty output — processing failed.');
    if (bytes[0] !== 0x25 || bytes[1] !== 0x50 ||
        bytes[2] !== 0x44 || bytes[3] !== 0x46) {
      throw new Error(
        ctx + ' output does not start with %PDF- header. ' +
        'First bytes: ' +
        Array.from(bytes.slice(0, 8))
          .map(function (b) { return ('0' + b.toString(16)).slice(-2); })
          .join(' ')
      );
    }
  }

  /* Wrap Uint8Array in a PDF Blob after validation */
  function _makePDFBlob(bytes, ctx) {
    _assertValidPDF(bytes, ctx);
    return new Blob([bytes], { type: 'application/pdf' });
  }

  /* ── File reader ────────────────────────────────────────── */
  function _readAB(file) {
    return new Promise(function (res, rej) {
      var r = new FileReader();
      r.onload  = function () { res(r.result); };
      r.onerror = function () { rej(new Error('Cannot read: ' + (file.name || 'file'))); };
      r.readAsArrayBuffer(file);
    });
  }

  /* ── Render one PDF page to canvas ─────────────────────── */
  function _renderPage(page, scale) {
    var vp     = page.getViewport({ scale: scale || 1.5 });
    var canvas = document.createElement('canvas');
    canvas.width  = Math.round(vp.width);
    canvas.height = Math.round(vp.height);
    var ctx    = canvas.getContext('2d');
    return page.render({ canvasContext: ctx, viewport: vp }).promise
      .then(function () { return { canvas: canvas, vp: vp }; });
  }

  /* ── Canvas → Blob (replaces unreliable fetch(dataURL)) ─── */
  function _canvasToBlob(canvas, type, quality) {
    return new Promise(function (resolve, reject) {
      try {
        canvas.toBlob(function (b) {
          if (b && b.size > 0) { resolve(b); return; }
          /* Fallback: dataURL → Blob */
          try {
            var du    = canvas.toDataURL(type || 'image/jpeg', quality || 0.85);
            var parts = du.split(',');
            var mime  = parts[0].match(/:(.*?);/)[1];
            var raw   = atob(parts[1]);
            var arr   = new Uint8Array(raw.length);
            for (var i = 0; i < raw.length; i++) arr[i] = raw.charCodeAt(i);
            resolve(new Blob([arr], { type: mime }));
          } catch (e2) { reject(e2); }
        }, type || 'image/jpeg', quality || 0.85);
      } catch (e) {
        reject(e);
      }
    });
  }

  /* ── HTML-escape helper ─────────────────────────────────── */
  function _esc(s) {
    return String(s)
      .replace(/&/g, '&amp;').replace(/</g, '&lt;')
      .replace(/>/g, '&gt;').replace(/"/g, '&quot;');
  }


  /* ═══════════════════════════════════════════════════════════
     1. Compress PDF
        Uses PDF.js to render each page to canvas at 1.5× scale,
        then re-assembles with jsPDF at the chosen JPEG quality.
     ═══════════════════════════════════════════════════════════ */
  function compressPDF(file, quality) {
    quality = (typeof quality === 'number' && quality > 0 && quality <= 1) ? quality : 0.75;

    return Promise.all([_getPdfjs(), _getJsPDF()])
      .then(function (libs) {
        var pdfjs = libs[0];
        var JsPDF = libs[1];

        return _readAB(file).then(function (ab) {
          return pdfjs.getDocument({ data: new Uint8Array(ab) }).promise
            .then(function (pdf) {
              if (pdf.numPages === 0) throw new Error('PDF contains no pages.');

              var jsdoc = null;
              var chain = Promise.resolve();

              for (var i = 1; i <= pdf.numPages; i++) {
                (function (n) {
                  chain = chain.then(function () {
                    return pdf.getPage(n).then(function (pg) {
                      return _renderPage(pg, 1.5).then(function (r) {
                        /* Convert pixel → point (72 dpi / 96 dpi = 0.75) */
                        var ptW = r.vp.width  * 0.75;
                        var ptH = r.vp.height * 0.75;
                        var ori = ptW >= ptH ? 'landscape' : 'portrait';

                        if (!jsdoc) {
                          /* Create document on first page */
                          jsdoc = new JsPDF({
                            orientation: ori,
                            unit:        'pt',
                            format:      [ptW, ptH],
                            compress:    true,
                          });
                        } else {
                          jsdoc.addPage([ptW, ptH], ori);
                        }

                        /* Use canvas.toDataURL directly — fastest and most reliable */
                        var imgData = r.canvas.toDataURL('image/jpeg', quality);
                        jsdoc.addImage(imgData, 'JPEG', 0, 0, ptW, ptH, '', 'FAST');
                      });
                    });
                  });
                })(i);
              }

              return chain.then(function () {
                if (!jsdoc) throw new Error('No pages were rendered during compression.');
                /* output('uint8array') is universally reliable in all jsPDF builds */
                var u8 = jsdoc.output('uint8array');
                return _makePDFBlob(u8, 'compressPDF');
              });
            });
        });
      });
  }


  /* ═══════════════════════════════════════════════════════════
     2. Merge PDFs
     ═══════════════════════════════════════════════════════════ */
  function mergePDFs(files) {
    if (!files || files.length < 2)
      return Promise.reject(new Error('Please upload at least 2 PDF files to merge.'));

    return _getPDFLib().then(function (PDFLib) {
      var PDFDocument = PDFLib.PDFDocument;
      return PDFDocument.create().then(function (merged) {
        var chain = Promise.resolve();
        files.forEach(function (file, idx) {
          chain = chain.then(function () {
            return _readAB(file).then(function (ab) {
              return PDFDocument.load(ab, { ignoreEncryption: true })
                .then(function (src) {
                  return merged.copyPages(src, src.getPageIndices())
                    .then(function (pages) {
                      if (!pages.length)
                        throw new Error('File ' + (idx + 1) + ' (' + file.name + ') has no pages.');
                      pages.forEach(function (p) { merged.addPage(p); });
                    });
                });
            });
          });
        });
        return chain.then(function () {
          var total = merged.getPageCount();
          if (total === 0) throw new Error('Merged document has no pages.');
          return merged.save().then(function (bytes) {
            return _makePDFBlob(bytes, 'mergePDFs');
          });
        });
      });
    });
  }


  /* ═══════════════════════════════════════════════════════════
     3. Split PDF
        splitType: 'all' (one page per file) | 'ranges' (custom)
        splitAt:   comma-separated pages/ranges e.g. "1,3-5,7"
     ═══════════════════════════════════════════════════════════ */
  function splitPDF(file, splitType, splitAt) {
    splitType = splitType || 'all';

    return _getPDFLib().then(function (PDFLib) {
      var PDFDocument = PDFLib.PDFDocument;
      return _readAB(file).then(function (ab) {
        return PDFDocument.load(ab, { ignoreEncryption: true })
          .then(function (src) {
            var total  = src.getPageCount();
            if (total === 0) throw new Error('PDF has no pages.');

            var ranges = [];

            if (splitType === 'all') {
              for (var i = 0; i < total; i++) ranges.push([i, i]);
            } else {
              (splitAt || '').split(',').forEach(function (r) {
                r = r.trim();
                if (!r) return;
                if (r.indexOf('-') !== -1) {
                  var p = r.split('-');
                  var s = Math.max(1, parseInt(p[0], 10)) - 1;
                  var e = Math.min(total, parseInt(p[1], 10)) - 1;
                  if (!isNaN(s) && !isNaN(e) && s <= e) ranges.push([s, e]);
                } else {
                  var n = parseInt(r, 10) - 1;
                  if (!isNaN(n) && n >= 0 && n < total) ranges.push([n, n]);
                }
              });
            }

            if (!ranges.length)
              throw new Error('No valid page ranges specified. Example: "1-3,5,7-9"');

            var blobs = [];
            var chain = Promise.resolve();

            ranges.forEach(function (range, ri) {
              chain = chain.then(function () {
                var indices = [];
                for (var j = range[0]; j <= range[1]; j++) indices.push(j);
                return PDFDocument.create().then(function (doc) {
                  return doc.copyPages(src, indices).then(function (pages) {
                    pages.forEach(function (p) { doc.addPage(p); });
                    return doc.save().then(function (bytes) {
                      var blob = _makePDFBlob(bytes, 'splitPDF range ' + ri);
                      var name = splitType === 'all'
                        ? 'page_' + (range[0] + 1) + '.pdf'
                        : 'pages_' + (range[0] + 1) + '_to_' + (range[1] + 1) + '.pdf';
                      blobs.push({ blob: blob, name: name });
                    });
                  });
                });
              });
            });

            return chain.then(function () {
              if (!blobs.length) throw new Error('Split produced no output files.');
              return blobs;
            });
          });
      });
    });
  }


  /* ═══════════════════════════════════════════════════════════
     4. Rotate PDF
        degrees: 90 | 180 | 270
        pages:   'all' | '1,3,5' (1-based)
     ═══════════════════════════════════════════════════════════ */
  function rotatePDF(file, degrees, pages) {
    degrees = parseInt(degrees, 10);
    if (![90, 180, 270, -90, -180, -270].includes(degrees))
      return Promise.reject(new Error('Rotation must be 90, 180 or 270 degrees.'));
    pages = pages || 'all';

    return _getPDFLib().then(function (PDFLib) {
      var PDFDocument = PDFLib.PDFDocument;
      var degFn       = PDFLib.degrees;
      return _readAB(file).then(function (ab) {
        return PDFDocument.load(ab, { ignoreEncryption: true })
          .then(function (doc) {
            var n = doc.getPageCount();
            if (n === 0) throw new Error('PDF has no pages.');

            var idxs;
            if (pages === 'all') {
              idxs = [];
              for (var i = 0; i < n; i++) idxs.push(i);
            } else {
              idxs = String(pages).split(',')
                .map(function (p) { return parseInt(p.trim(), 10) - 1; })
                .filter(function (i) { return i >= 0 && i < n; });
            }
            if (!idxs.length)
              throw new Error('No valid page numbers specified for rotation.');

            idxs.forEach(function (i) {
              var pg  = doc.getPage(i);
              var cur = pg.getRotation().angle;
              pg.setRotation(degFn(((cur + degrees) % 360 + 360) % 360));
            });

            return doc.save().then(function (bytes) {
              return _makePDFBlob(bytes, 'rotatePDF');
            });
          });
      });
    });
  }


  /* ═══════════════════════════════════════════════════════════
     5. Remove Pages
        pageList: [1,3,5] — 1-based page numbers
     ═══════════════════════════════════════════════════════════ */
  function removePages(file, pageList) {
    if (!pageList || !pageList.length)
      return Promise.reject(new Error('Specify at least one page number to remove.'));

    return _getPDFLib().then(function (PDFLib) {
      var PDFDocument = PDFLib.PDFDocument;
      return _readAB(file).then(function (ab) {
        return PDFDocument.load(ab, { ignoreEncryption: true })
          .then(function (src) {
            var total    = src.getPageCount();
            var toRemove = {};
            pageList.forEach(function (p) {
              var i = parseInt(p, 10);
              if (i >= 1 && i <= total) toRemove[i - 1] = true;
            });
            var keep = [];
            for (var i = 0; i < total; i++) { if (!toRemove[i]) keep.push(i); }
            if (!keep.length)
              throw new Error('Cannot remove all pages — document must have at least one page.');
            return PDFDocument.create().then(function (doc) {
              return doc.copyPages(src, keep).then(function (pages) {
                pages.forEach(function (p) { doc.addPage(p); });
                return doc.save().then(function (bytes) {
                  return _makePDFBlob(bytes, 'removePages');
                });
              });
            });
          });
      });
    });
  }


  /* ═══════════════════════════════════════════════════════════
     6. Extract Pages
        pageList: [2,4,6] — 1-based
     ═══════════════════════════════════════════════════════════ */
  function extractPages(file, pageList) {
    if (!pageList || !pageList.length)
      return Promise.reject(new Error('Specify at least one page number to extract.'));

    return _getPDFLib().then(function (PDFLib) {
      var PDFDocument = PDFLib.PDFDocument;
      return _readAB(file).then(function (ab) {
        return PDFDocument.load(ab, { ignoreEncryption: true })
          .then(function (src) {
            var total   = src.getPageCount();
            var indices = pageList
              .map(function (p) { return parseInt(p, 10) - 1; })
              .filter(function (i) { return i >= 0 && i < total; });
            if (!indices.length)
              throw new Error('None of the specified page numbers are valid (1–' + total + ').');
            return PDFDocument.create().then(function (doc) {
              return doc.copyPages(src, indices).then(function (pages) {
                pages.forEach(function (p) { doc.addPage(p); });
                return doc.save().then(function (bytes) {
                  return _makePDFBlob(bytes, 'extractPages');
                });
              });
            });
          });
      });
    });
  }


  /* ═══════════════════════════════════════════════════════════
     7. Reorder Pages
        newOrder: [3,1,2] — 1-based, new sequence
     ═══════════════════════════════════════════════════════════ */
  function reorderPages(file, newOrder) {
    if (!newOrder || !newOrder.length)
      return Promise.reject(new Error('Provide the new page order.'));

    return _getPDFLib().then(function (PDFLib) {
      var PDFDocument = PDFLib.PDFDocument;
      return _readAB(file).then(function (ab) {
        return PDFDocument.load(ab, { ignoreEncryption: true })
          .then(function (src) {
            var total   = src.getPageCount();
            var indices = newOrder
              .map(function (p) { return parseInt(p, 10) - 1; })
              .filter(function (i) { return i >= 0 && i < total; });
            if (!indices.length)
              throw new Error('No valid page numbers in the reorder list.');
            return PDFDocument.create().then(function (doc) {
              return doc.copyPages(src, indices).then(function (pages) {
                pages.forEach(function (p) { doc.addPage(p); });
                return doc.save().then(function (bytes) {
                  return _makePDFBlob(bytes, 'reorderPages');
                });
              });
            });
          });
      });
    });
  }


  /* ═══════════════════════════════════════════════════════════
     8. PDF → JPG
        dpi:     72–300 (default 150)
        quality: 0.1–1.0 (default 0.92)
        Returns: [{blob, name, dataURL}, ...]
     ═══════════════════════════════════════════════════════════ */
  function pdfToJPG(file, dpi, quality) {
    dpi     = Math.max(72, Math.min(300, dpi || 150));
    quality = Math.max(0.1, Math.min(1.0, quality || 0.92));
    var scale = dpi / 96;

    return _getPdfjs().then(function (pdfjs) {
      return _readAB(file).then(function (ab) {
        return pdfjs.getDocument({ data: new Uint8Array(ab) }).promise
          .then(function (pdf) {
            if (pdf.numPages === 0) throw new Error('PDF has no pages.');

            var images = [];
            var chain  = Promise.resolve();

            for (var i = 1; i <= pdf.numPages; i++) {
              (function (n) {
                chain = chain.then(function () {
                  return pdf.getPage(n).then(function (pg) {
                    return _renderPage(pg, scale).then(function (r) {
                      var dataURL = r.canvas.toDataURL('image/jpeg', quality);
                      return _canvasToBlob(r.canvas, 'image/jpeg', quality)
                        .then(function (blob) {
                          if (!blob || blob.size === 0)
                            throw new Error('Page ' + n + ' produced an empty image.');
                          images.push({
                            blob:    blob,
                            name:    'page_' + n + '.jpg',
                            dataURL: dataURL,
                          });
                        });
                    });
                  });
                });
              })(i);
            }

            return chain.then(function () {
              if (!images.length) throw new Error('No images were generated from the PDF.');
              return images;
            });
          });
      });
    });
  }


  /* ═══════════════════════════════════════════════════════════
     9. JPG → PDF
        files: File[] — JPEG, PNG, GIF, WebP images
     ═══════════════════════════════════════════════════════════ */
  function jpgToPDF(files) {
    if (!files || !files.length)
      return Promise.reject(new Error('No image files provided.'));

    return _getPDFLib().then(function (PDFLib) {
      var PDFDocument = PDFLib.PDFDocument;
      return PDFDocument.create().then(function (doc) {
        var chain = Promise.resolve();
        files.forEach(function (file) {
          chain = chain.then(function () {
            return _readAB(file).then(function (ab) {
              var bytes = new Uint8Array(ab);
              var name  = (file.name || '').toLowerCase();
              var isPng = file.type === 'image/png' || name.endsWith('.png');
              return (isPng ? doc.embedPng(bytes) : doc.embedJpg(bytes))
                .then(function (img) {
                  var pg = doc.addPage([img.width, img.height]);
                  pg.drawImage(img, { x: 0, y: 0, width: img.width, height: img.height });
                });
            });
          });
        });
        return chain.then(function () {
          return doc.save().then(function (bytes) {
            return _makePDFBlob(bytes, 'jpgToPDF');
          });
        });
      });
    });
  }


  /* ═══════════════════════════════════════════════════════════
     10. Add Watermark
         text:          watermark string
         opts.opacity:  0–1 (default 0.25)
         opts.fontSize: points (default 48)
         opts.rotation: degrees (default 45)
         opts.color:    [r,g,b] 0–1 (default grey)
     ═══════════════════════════════════════════════════════════ */
  function addWatermark(file, text, opts) {
    if (!text || !text.trim())
      return Promise.reject(new Error('Watermark text is required.'));

    opts = opts || {};
    var opacity  = opts.opacity  !== undefined ? opts.opacity  : 0.25;
    var fontSize = opts.fontSize !== undefined ? opts.fontSize : 48;
    var rotation = opts.rotation !== undefined ? opts.rotation : 45;
    var color    = opts.color    || [0.75, 0.75, 0.75];

    return _getPDFLib().then(function (PDFLib) {
      var PDFDocument   = PDFLib.PDFDocument;
      var rgb           = PDFLib.rgb;
      var StandardFonts = PDFLib.StandardFonts;
      var degFn         = PDFLib.degrees;

      return _readAB(file).then(function (ab) {
        return PDFDocument.load(ab, { ignoreEncryption: true })
          .then(function (doc) {
            return doc.embedFont(StandardFonts.HelveticaBold).then(function (font) {
              doc.getPages().forEach(function (pg) {
                var sz    = pg.getSize();
                var textW = font.widthOfTextAtSize(text, fontSize);
                pg.drawText(text, {
                  x:       (sz.width  - textW)   / 2,
                  y:       (sz.height - fontSize) / 2,
                  size:    fontSize,
                  font:    font,
                  color:   rgb(color[0], color[1], color[2]),
                  opacity: opacity,
                  rotate:  degFn(rotation),
                });
              });
              return doc.save().then(function (bytes) {
                return _makePDFBlob(bytes, 'addWatermark');
              });
            });
          });
      });
    });
  }


  /* ═══════════════════════════════════════════════════════════
     11. Lock PDF  (add password protection)
         userPwd:  password required to open
         ownerPwd: master password (auto-generated if omitted)

         Note: pdf-lib 1.x uses RC4-based PDF encryption (PDF 1.4
         standard). The resulting Blob will NOT start with %PDF-
         in its plain-text header because the content is encrypted.
         We intentionally skip _assertValidPDF here — the file IS
         a valid encrypted PDF that opens in any PDF reader.
     ═══════════════════════════════════════════════════════════ */
  function lockPDF(file, userPwd, ownerPwd) {
    if (!userPwd || !String(userPwd).trim())
      return Promise.reject(new Error('A user password is required to lock the PDF.'));

    ownerPwd = ownerPwd || (userPwd + '_own_' + Math.random().toString(36).slice(2, 9));

    return _getPDFLib().then(function (PDFLib) {
      var PDFDocument = PDFLib.PDFDocument;
      return _readAB(file).then(function (ab) {
        return PDFDocument.load(ab, { ignoreEncryption: true })
          .then(function (doc) {
            return doc.save({
              userPassword:  userPwd,
              ownerPassword: ownerPwd,
              permissions: {
                printing:             'highResolution',
                modifying:            false,
                copying:              false,
                annotating:           false,
                fillingForms:         false,
                contentAccessibility: false,
                documentAssembly:     false,
              },
            }).then(function (bytes) {
              /* Encrypted PDF bytes may not have a plain-text %PDF- start,
                 so we only check for non-empty output here. */
              if (!bytes || bytes.length === 0)
                throw new Error('lockPDF produced empty output.');
              return new Blob([bytes], { type: 'application/pdf' });
            });
          });
      });
    });
  }


  /* ═══════════════════════════════════════════════════════════
     12. Unlock PDF  (remove password)
         password: known password (optional — tries ignoreEncryption
         as fallback for ownership-locked-only PDFs)
     ═══════════════════════════════════════════════════════════ */
  function unlockPDF(file, password) {
    password = password || '';

    return _getPDFLib().then(function (PDFLib) {
      var PDFDocument = PDFLib.PDFDocument;
      return _readAB(file).then(function (ab) {
        var loadOpts = password
          ? { password: password }
          : { ignoreEncryption: true };

        return PDFDocument.load(ab, loadOpts)
          .catch(function () {
            /* Try the other mode as fallback */
            return PDFDocument.load(ab, { ignoreEncryption: true });
          })
          .then(function (doc) {
            return doc.save().then(function (bytes) {
              return _makePDFBlob(bytes, 'unlockPDF');
            });
          });
      });
    });
  }


  /* ═══════════════════════════════════════════════════════════
     13. PDF → Word (.doc)
         Extracts text via PDF.js, builds an MHT-style HTML Word
         document with proper Office XML namespaces.
         Returns a Blob with MIME type application/msword.
     ═══════════════════════════════════════════════════════════ */
  function pdfToWord(file, progressCb) {
    return _getPdfjs().then(function (pdfjs) {
      return _readAB(file).then(function (ab) {
        return pdfjs.getDocument({ data: new Uint8Array(ab) }).promise
          .then(function (pdf) {
            var n = pdf.numPages;
            if (n === 0) throw new Error('PDF has no pages.');

            var html = [
              '<!DOCTYPE html>',
              '<html xmlns:o="urn:schemas-microsoft-com:office:office"',
              '      xmlns:w="urn:schemas-microsoft-com:office:word"',
              '      xmlns="http://www.w3.org/TR/REC-html40">',
              '<head><meta charset="utf-8">',
              '<title>' + _esc(file.name.replace(/\.pdf$/i, '')) + '</title>',
              '<!--[if gte mso 9]>',
              '<xml><w:WordDocument><w:View>Print</w:View></w:WordDocument></xml>',
              '<![endif]-->',
              '<style>',
              'body{font-family:Calibri,Arial,sans-serif;font-size:11pt;',
              '     margin:2cm 2.5cm;line-height:1.6;color:#111}',
              '.page{margin-bottom:24pt;padding-bottom:12pt;',
              '      border-bottom:1px solid #bbb;page-break-after:always}',
              '.pg-label{font-size:8.5pt;color:#999;margin-bottom:8pt;',
              '          border-bottom:1px solid #eee;padding-bottom:2pt}',
              'p{margin:0 0 6pt}',
              '</style></head>',
              '<body>',
            ].join('\n');

            var chain = Promise.resolve();
            for (var i = 1; i <= n; i++) {
              (function (pg) {
                chain = chain.then(function () {
                  return pdf.getPage(pg).then(function (page) {
                    return page.getTextContent().then(function (tc) {
                      html += '\n<div class="page">\n';
                      html += '<div class="pg-label">— Page ' + pg + ' of ' + n + ' —</div>\n';

                      /* Group text items by Y coordinate (4-pt buckets) */
                      var rows = {};
                      tc.items.forEach(function (it) {
                        if (!it.str || !it.str.trim()) return;
                        var y = Math.round((it.transform ? it.transform[5] : 0) / 4) * 4;
                        if (!rows[y]) rows[y] = [];
                        rows[y].push(it.str);
                      });

                      /* Output lines sorted top-to-bottom (descending Y) */
                      Object.keys(rows)
                        .sort(function (a, b) { return Number(b) - Number(a); })
                        .forEach(function (y) {
                          var line = rows[y].join(' ').trim();
                          if (line) html += '<p>' + _esc(line) + '</p>\n';
                        });

                      html += '</div>\n';
                      if (progressCb) progressCb(Math.round((pg / n) * 90));
                    });
                  });
                });
              })(i);
            }

            return chain.then(function () {
              html += '\n</body>\n</html>';
              if (progressCb) progressCb(100);
              var blob = new Blob([html], { type: 'application/msword' });
              if (blob.size === 0) throw new Error('pdfToWord produced empty output.');
              return blob;
            });
          });
      });
    });
  }


  /* ═══════════════════════════════════════════════════════════
     14. Word → PDF
         Converts .doc/.docx using mammoth (HTML extraction)
         then renders to A4 PDF with jsPDF.
     ═══════════════════════════════════════════════════════════ */
  function wordToPDF(file, progressCb) {
    return _ensureLib('mammoth').then(function () {
      return _readAB(file).then(function (ab) {
        if (progressCb) progressCb(20);
        return mammoth.convertToHtml({ arrayBuffer: ab })
          .then(function (result) {
            if (progressCb) progressCb(55);

            /* Strip HTML, decode entities → plain text */
            var text = result.value
              .replace(/<\/p>/gi,     '\n')
              .replace(/<br\s*\/?>/gi, '\n')
              .replace(/<\/h[1-6]>/gi, '\n\n')
              .replace(/<[^>]+>/g,    '')
              .replace(/&amp;/g,  '&').replace(/&lt;/g,  '<')
              .replace(/&gt;/g,   '>').replace(/&nbsp;/g, ' ')
              .replace(/&quot;/g, '"').replace(/&#39;/g,  "'")
              .replace(/\r\n/g,   '\n')
              .trim();

            if (!text)
              throw new Error('No readable text found in the Word document.');

            return _getJsPDF().then(function (JsPDF) {
              var doc = new JsPDF({
                compress:    true,
                unit:        'mm',
                format:      'a4',
                orientation: 'portrait',
              });

              doc.setFont('helvetica', 'normal');
              doc.setFontSize(11);

              var y      = 18;
              var pageH  = doc.internal.pageSize.getHeight();
              var margin = 15;
              var lineH  = 7;
              var maxW   = 180;

              text.split(/\n+/).forEach(function (para) {
                para = para.trim();
                if (!para) { y += lineH * 0.4; return; }
                var lines = doc.splitTextToSize(para, maxW);
                lines.forEach(function (line) {
                  if (y + lineH > pageH - margin) { doc.addPage(); y = margin; }
                  doc.text(line, margin, y);
                  y += lineH;
                });
                y += 2;
              });

              if (progressCb) progressCb(95);
              var u8   = doc.output('uint8array');
              var blob = _makePDFBlob(u8, 'wordToPDF');
              if (progressCb) progressCb(100);
              return blob;
            });
          });
      });
    });
  }


  /* ═══════════════════════════════════════════════════════════
     15. Get page count
     ═══════════════════════════════════════════════════════════ */
  function getPageCount(file) {
    return _getPDFLib().then(function (PDFLib) {
      return _readAB(file).then(function (ab) {
        return PDFLib.PDFDocument.load(ab, { ignoreEncryption: true })
          .then(function (doc) { return doc.getPageCount(); });
      });
    });
  }


  /* ═══════════════════════════════════════════════════════════
     16. Extract plain text
         progressCb(0-100) called per page
     ═══════════════════════════════════════════════════════════ */
  function extractText(file, progressCb) {
    return _getPdfjs().then(function (pdfjs) {
      return _readAB(file).then(function (ab) {
        return pdfjs.getDocument({ data: new Uint8Array(ab) }).promise
          .then(function (pdf) {
            var n     = pdf.numPages;
            var text  = '';
            var chain = Promise.resolve();

            for (var i = 1; i <= n; i++) {
              (function (pg) {
                chain = chain.then(function () {
                  return pdf.getPage(pg).then(function (page) {
                    return page.getTextContent().then(function (tc) {
                      var line = tc.items.map(function (it) { return it.str; }).join(' ');
                      text += line + '\n\n';
                      if (progressCb) progressCb(Math.round((pg / n) * 100));
                    });
                  });
                });
              })(i);
            }

            return chain.then(function () {
              if (!text.trim())
                throw new Error('No selectable text found. This PDF may be image-based (scanned).');
              return text;
            });
          });
      });
    });
  }


  /* ── Public API ─────────────────────────────────────────── */
  return {
    compressPDF:  compressPDF,
    mergePDFs:    mergePDFs,
    splitPDF:     splitPDF,
    rotatePDF:    rotatePDF,
    removePages:  removePages,
    extractPages: extractPages,
    reorderPages: reorderPages,
    pdfToJPG:     pdfToJPG,
    jpgToPDF:     jpgToPDF,
    addWatermark: addWatermark,
    lockPDF:      lockPDF,
    unlockPDF:    unlockPDF,
    pdfToWord:    pdfToWord,
    wordToPDF:    wordToPDF,
    getPageCount: getPageCount,
    extractText:  extractText,
  };

})();
