/* ============================================================
   PDF Pro — Main JS  v5.0  (PRODUCTION-FIXED)
   ============================================================
   ROOT-CAUSE FIXES:
   -----------------
   FIX 1 — validateFile() enforced .pdf extension check
            strictly; non-PDF files cannot reach tool functions.

   FIX 2 — showToast() double-RAF trick ensures CSS transition
            fires even on first call (some browsers skip one RAF).

   FIX 3 — initDropZone: file input value reset BEFORE onFiles
            callback so same file can be re-selected immediately.

   FIX 4 — ProgressBar.finish() no longer auto-hides so the
            "100% Complete" state is visible when result appears.
            Tool pages call _prog.hide() themselves after result.

   Depends on: download-manager.js (loaded before this file)
   ============================================================ */
(function () {
  'use strict';

  /* ── Theme ─────────────────────────────────────────────── */
  var THEME_KEY    = 'pdfpro-theme';
  var currentTheme = localStorage.getItem(THEME_KEY) || 'dark';

  function applyTheme(t) {
    currentTheme = t;
    document.documentElement.setAttribute('data-theme', t);
    localStorage.setItem(THEME_KEY, t);
    document.querySelectorAll('.theme-toggle').forEach(function (btn) {
      btn.textContent = t === 'dark' ? '🌙' : '☀️';
      btn.setAttribute('aria-label',
        t === 'dark' ? 'Switch to light mode' : 'Switch to dark mode');
    });
  }
  applyTheme(currentTheme);

  document.addEventListener('click', function (e) {
    if (e.target.closest && e.target.closest('.theme-toggle')) {
      applyTheme(currentTheme === 'dark' ? 'light' : 'dark');
    }
  });

  /* ── DOM ready ─────────────────────────────────────────── */
  document.addEventListener('DOMContentLoaded', function () {

    /* Mobile nav */
    var menu   = document.querySelector('.nav-menu');
    var burger = document.querySelector('.hamburger');
    if (burger && menu) {
      burger.addEventListener('click', function () {
        var open = menu.classList.toggle('open');
        burger.classList.toggle('active', open);
        burger.setAttribute('aria-expanded', String(open));
      });
      menu.querySelectorAll('a').forEach(function (a) {
        a.addEventListener('click', function () {
          menu.classList.remove('open');
          burger.classList.remove('active');
          burger.setAttribute('aria-expanded', 'false');
        });
      });
    }

    /* Active nav link */
    var path = location.pathname.split('/').pop() || 'index.html';
    document.querySelectorAll('.nav-menu a').forEach(function (a) {
      var href = a.getAttribute('href') || '';
      if (href === path || href.endsWith('/' + path) ||
          (path === '' && (href === 'index.html' || href === './'))) {
        a.classList.add('active');
      }
    });

    /* Scroll-reveal */
    if ('IntersectionObserver' in window) {
      var obs = new IntersectionObserver(function (entries) {
        entries.forEach(function (en) {
          if (en.isIntersecting) {
            en.target.classList.add('animate-in');
            obs.unobserve(en.target);
          }
        });
      }, { threshold: 0.07 });
      document.querySelectorAll('.tool-card, .feature-card, .blog-card')
        .forEach(function (el) { obs.observe(el); });
    }

    /* FAQ accordion */
    document.querySelectorAll('.faq-q').forEach(function (btn) {
      btn.addEventListener('click', function () {
        var item = btn.closest('.faq-item');
        var open = item.classList.toggle('open');
        btn.setAttribute('aria-expanded', String(open));
      });
    });

    /* Smooth anchor scroll */
    document.querySelectorAll('a[href^="#"]').forEach(function (a) {
      a.addEventListener('click', function (e) {
        var target = document.querySelector(a.getAttribute('href'));
        if (target) { e.preventDefault(); target.scrollIntoView({ behavior: 'smooth' }); }
      });
    });

    /* Contact form */
    var cForm = document.getElementById('contactForm');
    if (cForm) {
      cForm.addEventListener('submit', function (e) {
        e.preventDefault();
        showToast("Message sent! We'll reply within 24 hours.", 'success');
        cForm.reset();
      });
    }
  });

  /* ──────────────────────────────────────────────────────────
     TOAST SYSTEM  →  window.showToast(msg, type, duration)
     ────────────────────────────────────────────────────────── */
  window.showToast = function (msg, type, duration) {
    type     = type     || 'info';
    duration = duration || 4000;

    var container = document.querySelector('.toast-container');
    if (!container) {
      container = document.createElement('div');
      container.className = 'toast-container';
      container.setAttribute('role', 'alert');
      container.setAttribute('aria-live', 'polite');
      document.body.appendChild(container);
    }

    var icons = { success: '✅', error: '❌', info: 'ℹ️', warning: '⚠️' };
    var toast = document.createElement('div');
    toast.className = 'toast toast-' + type;
    toast.innerHTML =
      '<span class="toast-icon">'  + (icons[type] || 'ℹ️') + '</span>' +
      '<span class="toast-msg">'   + _escHtml(msg) + '</span>' +
      '<button class="toast-close" aria-label="Dismiss">✕</button>';
    toast.querySelector('.toast-close').addEventListener('click', function () {
      _removeToast(toast);
    });
    container.appendChild(toast);
    /* Double RAF ensures CSS transition fires on first render */
    requestAnimationFrame(function () {
      requestAnimationFrame(function () { toast.classList.add('show'); });
    });
    setTimeout(function () { _removeToast(toast); }, duration);
  };

  function _removeToast(t) {
    t.classList.remove('show');
    setTimeout(function () { if (t.parentNode) t.parentNode.removeChild(t); }, 400);
  }

  /* ──────────────────────────────────────────────────────────
     FORMAT BYTES  →  window.formatBytes(bytes)
     ────────────────────────────────────────────────────────── */
  window.formatBytes = function (bytes, dp) {
    if (!bytes) return '0 B';
    dp = (dp !== undefined) ? dp : 1;
    var k  = 1024;
    var sz = ['B', 'KB', 'MB', 'GB'];
    var i  = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(dp)) + '\u202f' + sz[i];
  };

  /* ──────────────────────────────────────────────────────────
     FILE VALIDATION  →  window.validateFile(file, type)
     type: 'pdf' | 'image' | 'word'
     Returns true if valid; shows error toast + returns false if not.
     ────────────────────────────────────────────────────────── */
  window.validateFile = function (file, type) {
    if (!file) { showToast('No file selected.', 'error'); return false; }

    /* 500 MB cap */
    if (file.size > 500 * 1024 * 1024) {
      showToast(
        'File too large (' + formatBytes(file.size) + '). Maximum is 500\u202fMB.',
        'error', 6000);
      return false;
    }

    var name = (file.name || '').toLowerCase();
    var mime = (file.type || '').toLowerCase();
    var ext  = name.split('.').pop();

    if (type === 'pdf') {
      if (ext !== 'pdf') {
        showToast(
          'Only PDF files are accepted. You uploaded: ' + file.name,
          'error', 5000);
        return false;
      }
      /* Warn if MIME says otherwise (may be misnamed) */
      if (mime && mime !== 'application/pdf' && mime !== '' && mime !== 'application/octet-stream') {
        showToast(
          'Warning: file has .pdf extension but reports type "' + mime +
          '". Attempting to process anyway.',
          'warning', 5000);
      }
    } else if (type === 'image') {
      var imgExts  = ['jpg','jpeg','png','gif','webp','bmp','tiff','tif'];
      if (!imgExts.includes(ext)) {
        showToast(
          'Only image files (JPG, PNG, GIF, WebP) are accepted. You uploaded: ' + file.name,
          'error', 5000);
        return false;
      }
    } else if (type === 'word') {
      if (!['doc','docx'].includes(ext)) {
        showToast(
          'Only Word files (.doc, .docx) are accepted. You uploaded: ' + file.name,
          'error', 5000);
        return false;
      }
    }
    return true;
  };

  /* ──────────────────────────────────────────────────────────
     FILE READERS  →  window.readFileAsArrayBuffer / DataURL
     ────────────────────────────────────────────────────────── */
  window.readFileAsArrayBuffer = function (file) {
    return new Promise(function (res, rej) {
      var r = new FileReader();
      r.onload  = function () { res(r.result); };
      r.onerror = function () { rej(new Error('Cannot read: ' + file.name)); };
      r.readAsArrayBuffer(file);
    });
  };

  window.readFileAsDataURL = function (file) {
    return new Promise(function (res, rej) {
      var r = new FileReader();
      r.onload  = function () { res(r.result); };
      r.onerror = function () { rej(new Error('Cannot read: ' + file.name)); };
      r.readAsDataURL(file);
    });
  };

  /* ──────────────────────────────────────────────────────────
     PROGRESS BAR  →  window.ProgressBar(selector)
     ────────────────────────────────────────────────────────── */
  window.ProgressBar = function (selector) {
    var wrap = document.querySelector(selector);
    var noop = function () {};
    if (!wrap) return { show: noop, set: noop, finish: noop, hide: noop };

    var fill = wrap.querySelector('.progress-fill');
    var pct  = wrap.querySelector('.progress-pct');
    var stat = wrap.querySelector('.progress-status');

    function _set(v, msg) {
      var p = Math.min(100, Math.max(0, Math.round(v)));
      if (fill) fill.style.width = p + '%';
      if (pct)  pct.textContent  = p + '%';
      if (msg && stat) stat.textContent = msg;
    }

    return {
      show:   function (msg) {
        wrap.classList.add('active');
        if (stat) stat.textContent = msg || 'Processing…';
        _set(5);
      },
      set:    function (v, msg) { _set(v, msg); },
      finish: function (msg) {
        /* FIX: do NOT auto-hide — let tool page control visibility */
        _set(100, msg || 'Complete ✓');
      },
      hide:   function () {
        wrap.classList.remove('active');
        _set(0);
        if (stat) stat.textContent = 'Processing…';
      },
    };
  };

  /* ──────────────────────────────────────────────────────────
     DROP ZONE  →  window.initDropZone(el, input, opts)
     ────────────────────────────────────────────────────────── */
  window.initDropZone = function (zoneEl, fileInput, opts) {
    if (!zoneEl) return;
    opts     = opts     || {};
    var fileType = opts.fileType || 'pdf';
    var multiple = !!opts.multiple;
    var onFiles  = opts.onFiles;

    /* Drag feedback */
    ['dragenter','dragover'].forEach(function (ev) {
      zoneEl.addEventListener(ev, function (e) {
        e.preventDefault(); e.stopPropagation();
        zoneEl.classList.add('drag-over');
      });
    });
    ['dragleave','dragend'].forEach(function (ev) {
      zoneEl.addEventListener(ev, function () {
        zoneEl.classList.remove('drag-over');
      });
    });
    zoneEl.addEventListener('drop', function (e) {
      e.preventDefault(); e.stopPropagation();
      zoneEl.classList.remove('drag-over');
      _handle(e.dataTransfer.files);
    });

    /* File picker */
    if (fileInput) {
      fileInput.addEventListener('change', function () {
        if (fileInput.files && fileInput.files.length) {
          var files = fileInput.files;
          /* FIX: reset value BEFORE callback so same file can be re-picked */
          var arr = Array.from(files);
          fileInput.value = '';
          _handle({ length: arr.length, item: function(i){return arr[i];}, [Symbol.iterator]: function*(){yield*arr;} });
        }
      });
    }

    /* Click / keyboard opens picker */
    zoneEl.setAttribute('tabindex', '0');
    zoneEl.addEventListener('click', function (e) {
      if (fileInput && !e.target.closest('input')) fileInput.click();
    });
    zoneEl.addEventListener('keydown', function (e) {
      if ((e.key === 'Enter' || e.key === ' ') && fileInput) {
        e.preventDefault(); fileInput.click();
      }
    });

    function _handle(fileList) {
      if (!fileList || !fileList.length) return;
      var arr   = Array.from(fileList);
      var valid = arr.filter(function (f) { return validateFile(f, fileType); });
      if (!valid.length) return;
      if (onFiles) onFiles(multiple ? valid : [valid[0]]);
    }
  };

  /* ── HTML-escape ────────────────────────────────────────── */
  function _escHtml(s) {
    return String(s)
      .replace(/&/g,'&amp;').replace(/</g,'&lt;')
      .replace(/>/g,'&gt;').replace(/"/g,'&quot;');
  }

})();
