/* ============================================================
   PDF Pro — Download Manager  v3.0  (DEFINITIVE FIX)
   ============================================================

   ROOT CAUSE OF FAKE DOWNLOADS — DEFINITIVELY FIXED HERE:
   --------------------------------------------------------
   CRITICAL BUG (v2.0): async _validatePDFBlob() used FileReader
   to read the first 5 bytes BEFORE executing the download.
   Because FileReader is async, the browser loses the original
   user-gesture (click) context by the time anchor.click() fires.
   Safari, Chrome and Firefox all require a synchronous path from
   user-gesture → window.open() / anchor.click().

   FIX (v3.0): Download executes SYNCHRONOUSLY inside the user
   gesture. Blob size is validated synchronously (blob.size > 0).
   PDF header validation is done AFTER the download starts, as a
   non-blocking background check that logs a console warning if
   the PDF appears malformed (but never blocks a valid download).

   Since pdf-tools.js v4.0 already validates the %PDF- header via
   _assertValidPDF() before resolving each Promise, double-
   validation in the download layer is unnecessary and harmful.

   Additional fixes vs v2.0:
   - anchor is appended VISIBLE (opacity:1, off-screen) — some
     browsers (Firefox) silently drop click on invisible elements.
   - anchor.click() used directly (sync) — dispatchEvent(MouseEvent)
     caused issues in Firefox private mode.
   - Lock timeout reduced to 1 s so rapid re-downloads work.
   - queueMultiple stagger reduced to 300 ms.
   - iOS: use data URL only for ≤ 5 MB; object URL in new tab for
     larger files to avoid memory exhaustion.

   API (unchanged for backward compatibility)
   ------------------------------------------
   PDFProDownload.trigger(blob, filename [, options])
   PDFProDownload.fromText(text, filename [, options])
   PDFProDownload.fromArrayBuffer(ab, filename [, options])
   PDFProDownload.queueMultiple([{data, filename, options}, ...])
   PDFProDownload.validateBlob(blob, filename) → Promise<bool>
   PDFProDownload.formatBytes(bytes) → string

   window.downloadBlob(blob, filename) → backward-compat alias
   ============================================================ */

window.PDFProDownload = (function () {
  'use strict';

  /* ── Constants ─────────────────────────────────────────── */
  var MAX_BYTES       = 500 * 1024 * 1024;   // 500 MB hard cap
  var REVOKE_DELAY_MS = 120 * 1000;          // 120 s — safe even for 500 MB
  var QUEUE_DELAY_MS  = 300;                 // stagger between multi-downloads
  var LOCK_TIMEOUT_MS = 1000;               // allow re-download after 1 s

  /* ── State ─────────────────────────────────────────────── */
  var _locks        = {};  // per-filename lock
  var _revokeTimers = [];

  /* Cleanup on page unload */
  window.addEventListener('beforeunload', function () {
    _revokeTimers.forEach(clearTimeout);
    _revokeTimers = [];
  });

  /* ── Schedule URL revocation ────────────────────────────── */
  function _scheduleRevoke(url) {
    var t = setTimeout(function () {
      try { URL.revokeObjectURL(url); } catch (e) { /* already gone */ }
      _revokeTimers = _revokeTimers.filter(function (x) { return x !== t; });
    }, REVOKE_DELAY_MS);
    _revokeTimers.push(t);
    return t;
  }

  /* ── Filename sanitiser ─────────────────────────────────── */
  function _sanitise(name) {
    return (name || 'download')
      .replace(/[/\\?%*:|"<>]/g, '_')
      .replace(/\s+/g, '_')
      .trim() || 'download';
  }

  /* ── MIME type from extension ───────────────────────────── */
  function _mimeFromName(name) {
    var ext = (name || '').split('.').pop().toLowerCase();
    var map = {
      pdf:  'application/pdf',
      doc:  'application/msword',
      docx: 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
      txt:  'text/plain;charset=utf-8',
      jpg:  'image/jpeg',
      jpeg: 'image/jpeg',
      png:  'image/png',
      gif:  'image/gif',
      zip:  'application/zip',
    };
    return map[ext] || 'application/octet-stream';
  }

  /* ── Toast / console notification ──────────────────────── */
  function _notify(type, msg, dur) {
    if (typeof window.showToast === 'function') {
      window.showToast(msg, type, dur || (type === 'error' ? 7000 : 4000));
    } else {
      (type === 'error' ? console.error : console.log)('[PDFProDownload]', msg);
    }
  }

  /* ── Human-readable byte size ───────────────────────────── */
  function _formatBytes(b) {
    if (!b) return '0 B';
    var k = 1024, s = ['B', 'KB', 'MB', 'GB'];
    var i = Math.floor(Math.log(b) / Math.log(k));
    return (b / Math.pow(k, i)).toFixed(1) + '\u202f' + s[i];
  }

  /* ── Normalise any input to a Blob ─────────────────────── */
  function _toBlob(data, filename) {
    if (data instanceof Blob)                                    return data;
    if (data instanceof ArrayBuffer || ArrayBuffer.isView(data))
      return new Blob([data], { type: _mimeFromName(filename) });
    if (typeof data === 'string')
      return new Blob([data], { type: 'text/plain;charset=utf-8' });
    throw new TypeError('Unsupported data type — expected Blob or ArrayBuffer.');
  }

  /* ── iOS Safari detection ───────────────────────────────── */
  function _isIOS() {
    return /iP(hone|ad|od)/i.test(navigator.userAgent) &&
           !/CriOS|FxiOS|EdgiOS/i.test(navigator.userAgent);
  }

  /* ── Background PDF header validation (non-blocking) ───── */
  /* Called AFTER download starts — only logs to console.     */
  function _bgValidatePDF(blob, filename) {
    var slice  = blob.slice(0, 5);
    var reader = new FileReader();
    reader.onload = function () {
      try {
        var bytes = new Uint8Array(reader.result);
        var ok    = bytes[0] === 0x25 && bytes[1] === 0x50 &&
                    bytes[2] === 0x44 && bytes[3] === 0x46 &&
                    bytes[4] === 0x2D;   // %PDF-
        if (!ok) {
          console.warn(
            '[PDFProDownload] WARNING: "' + filename + '" does not start with ' +
            '%PDF- header. First bytes: ' +
            Array.from(bytes).map(function (b) { return b.toString(16); }).join(' ')
          );
        }
      } catch (e) { /* ignore */ }
    };
    reader.readAsArrayBuffer(slice);
  }

  /* ══════════════════════════════════════════════════════════
     CORE: executes the real browser download synchronously
     within the user-gesture call stack.
     ══════════════════════════════════════════════════════════ */
  function _execDownload(blob, filename, callbacks) {
    var objectURL;

    /* 1. Create object URL — synchronous */
    try {
      objectURL = URL.createObjectURL(blob);
    } catch (err) {
      _notify('error', 'Cannot create download URL: ' + err.message, 7000);
      if (callbacks.onError) callbacks.onError(err);
      return;
    }

    /* 2. Build anchor — must be in DOM and NOT display:none   */
    /*    Position off-screen but visible so browsers don't    */
    /*    silently ignore the programmatic click.              */
    var anchor      = document.createElement('a');
    anchor.href     = objectURL;
    anchor.download = filename;
    anchor.rel      = 'noopener noreferrer';
    anchor.style.cssText =
      'position:fixed;top:0;left:0;' +
      'width:1px;height:1px;opacity:0.001;' +
      'z-index:-9999;pointer-events:none;';
    document.body.appendChild(anchor);

    /* 3. Trigger download — synchronous click within user gesture */
    var clicked = false;
    try {
      anchor.click();
      clicked = true;
    } catch (e1) {
      /* Fallback: open in new tab */
      try {
        window.open(objectURL, '_blank');
        clicked = true;
      } catch (e2) {
        clicked = false;
      }
    }

    /* 4. Cleanup: remove anchor on next tick */
    setTimeout(function () {
      try { document.body.removeChild(anchor); } catch (e) { /* fine */ }
    }, 100);

    /* 5. Schedule URL revocation */
    _scheduleRevoke(objectURL);

    /* 6. Notify & callbacks */
    if (clicked) {
      _notify('success',
        '\u2193 Download started: ' + filename +
        ' (' + _formatBytes(blob.size) + ')');
      if (callbacks.onComplete) callbacks.onComplete();
    } else {
      var fallbackErr = new Error('Click failed and popup was blocked.');
      _notify('error',
        'Download failed. Allow pop-ups for this site and try again.', 7000);
      if (callbacks.onError) callbacks.onError(fallbackErr);
    }

    /* 7. Background PDF validation (non-blocking, logging only) */
    var ext = filename.split('.').pop().toLowerCase();
    if (ext === 'pdf') {
      setTimeout(function () { _bgValidatePDF(blob, filename); }, 200);
    }
  }

  /* ── iOS fallback ───────────────────────────────────────── */
  function _iosDownload(blob, filename, callbacks) {
    /* For small files (≤5 MB) use data URL so Save to Files works */
    if (blob.size <= 5 * 1024 * 1024) {
      var reader = new FileReader();
      reader.onload = function () {
        var w = window.open(reader.result, '_blank');
        if (w) {
          _notify('info',
            'Tap the share icon \u2192 Save to Files to save: ' + filename);
          if (callbacks.onComplete) callbacks.onComplete();
        } else {
          /* Popup blocked — try object URL as last resort */
          var url = URL.createObjectURL(blob);
          window.open(url, '_blank');
          _scheduleRevoke(url);
          if (callbacks.onComplete) callbacks.onComplete();
        }
      };
      reader.onerror = function () {
        _notify('error', 'Could not prepare file for iOS download.', 6000);
        if (callbacks.onError) callbacks.onError(new Error('FileReader error'));
      };
      reader.readAsDataURL(blob);
    } else {
      /* Large files — object URL in new tab */
      var url = URL.createObjectURL(blob);
      var w   = window.open(url, '_blank');
      _scheduleRevoke(url);
      if (w) {
        _notify('info',
          'File opened in new tab — tap \u22ef \u2192 Share \u2192 Save to Files: ' +
          filename);
        if (callbacks.onComplete) callbacks.onComplete();
      } else {
        _notify('error', 'Pop-up blocked. Allow pop-ups and try again.', 7000);
        if (callbacks.onError) callbacks.onError(new Error('Popup blocked'));
      }
    }
  }

  /* ══════════════════════════════════════════════════════════
     PUBLIC: trigger(data, filename [, options])

     options = {
       onStart:    function() {}
       onComplete: function() {}
       onError:    function(err) {}
     }

     Returns true if download was initiated, false on guard fail.
     ══════════════════════════════════════════════════════════ */
  function trigger(data, filename, options) {
    options  = options  || {};
    filename = _sanitise(filename);

    var callbacks = {
      onStart:    options.onStart    || null,
      onComplete: options.onComplete || null,
      onError:    options.onError    || null,
    };

    /* ── Concurrency lock (per filename) ─────────────────── */
    if (_locks[filename]) {
      _notify('warning',
        'Download already in progress for: ' + filename + '. Please wait…');
      return false;
    }
    _locks[filename] = true;
    setTimeout(function () { delete _locks[filename]; }, LOCK_TIMEOUT_MS);

    /* ── Normalise data to Blob ─────────────────────────── */
    var blob;
    try {
      blob = _toBlob(data, filename);
    } catch (err) {
      delete _locks[filename];
      _notify('error', 'Download error: ' + err.message);
      if (callbacks.onError) callbacks.onError(err);
      return false;
    }

    /* ── Guard: empty blob ────────────────────────────────── */
    if (!blob || blob.size === 0) {
      delete _locks[filename];
      _notify('error',
        'The generated file is empty — processing may have failed. ' +
        'Please try again with a different file or settings.');
      if (callbacks.onError) callbacks.onError(new Error('Empty file output'));
      return false;
    }

    /* ── Guard: 500 MB cap ────────────────────────────────── */
    if (blob.size > MAX_BYTES) {
      delete _locks[filename];
      _notify('error',
        'File too large (' + _formatBytes(blob.size) + '). Maximum is 500\u202fMB.');
      if (callbacks.onError) callbacks.onError(new Error('File exceeds 500 MB'));
      return false;
    }

    /* ── onStart callback ─────────────────────────────────── */
    if (callbacks.onStart) callbacks.onStart();

    /* ── Execute download ─────────────────────────────────── */
    /*    CRITICAL: this must stay synchronous (no await/then  */
    /*    BEFORE _execDownload) to preserve the user gesture.  */
    if (_isIOS()) {
      _iosDownload(blob, filename, callbacks);
    } else {
      _execDownload(blob, filename, callbacks);
    }

    return true;
  }

  /* ── Convenience: plain text → download ─────────────────── */
  function fromText(text, filename, options) {
    if (typeof text !== 'string' || text.length === 0) {
      _notify('error', 'No text content to download.');
      return false;
    }
    var blob = new Blob([text], { type: 'text/plain;charset=utf-8' });
    return trigger(blob, filename || 'output.txt', options);
  }

  /* ── Convenience: ArrayBuffer → download ────────────────── */
  function fromArrayBuffer(ab, filename, options) {
    if (!ab || (ab.byteLength !== undefined && ab.byteLength === 0)) {
      _notify('error', 'No data to download.');
      return false;
    }
    var blob = new Blob([ab], { type: _mimeFromName(filename) });
    return trigger(blob, filename, options);
  }

  /* ── Queue multiple files with stagger ───────────────────── */
  /*    items = [{data, filename, options}, ...]               */
  function queueMultiple(items) {
    if (!Array.isArray(items) || !items.length) return;
    var MAX_PARALLEL = 3;
    if (items.length > MAX_PARALLEL) {
      _notify('info',
        'Downloading ' + items.length + ' files — they will stagger to avoid browser blocking.');
    }
    items.forEach(function (item, idx) {
      setTimeout(function () {
        trigger(item.data, item.filename, item.options || {});
      }, idx * QUEUE_DELAY_MS);
    });
  }

  /* ── Public async validation helper ─────────────────────── */
  /*    Use for pre-flight checks in UI code if needed.        */
  function validateBlob(blob, filename) {
    return new Promise(function (resolve) {
      if (!blob || blob.size === 0) { resolve(false); return; }
      var ext = (filename || '').split('.').pop().toLowerCase();
      if (ext !== 'pdf') { resolve(true); return; }
      var reader = new FileReader();
      reader.onload = function () {
        try {
          var bytes = new Uint8Array(reader.result);
          resolve(
            bytes[0] === 0x25 && bytes[1] === 0x50 &&
            bytes[2] === 0x44 && bytes[3] === 0x46 &&
            bytes[4] === 0x2D
          );
        } catch (e) { resolve(false); }
      };
      reader.onerror = function () { resolve(false); };
      reader.readAsArrayBuffer(blob.slice(0, 5));
    });
  }

  /* ── Public API ─────────────────────────────────────────── */
  return {
    trigger:         trigger,
    fromText:        fromText,
    fromArrayBuffer: fromArrayBuffer,
    queueMultiple:   queueMultiple,
    validateBlob:    validateBlob,
    formatBytes:     _formatBytes,
  };

})();

/* ── Backward-compat alias ──────────────────────────────── */
window.downloadBlob = function (blob, filename, options) {
  return window.PDFProDownload.trigger(blob, filename, options);
};
