<?php
/**
 * PDF Pro - API Proxy
 * Optional: Enables server-side API calls when CORS blocks browser requests.
 * Only needed if you enable APIs that don't support browser CORS.
 */

// Security: Only allow requests from same origin
$origin = $_SERVER['HTTP_ORIGIN'] ?? $_SERVER['HTTP_HOST'] ?? '';
$allowed = ['yourdomain.com', 'www.yourdomain.com', 'localhost'];
$host = parse_url($origin, PHP_URL_HOST) ?: $origin;
if (!in_array($host, $allowed)) {
    http_response_code(403);
    echo json_encode(['error' => 'Forbidden']);
    exit;
}

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: ' . ($origin ?: '*'));
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

$input = json_decode(file_get_contents('php://input'), true);
if (!$input || !isset($input['url'])) {
    http_response_code(400);
    echo json_encode(['error' => 'Missing url parameter']);
    exit;
}

$url     = $input['url'];
$method  = strtoupper($input['method'] ?? 'POST');
$headers = $input['headers'] ?? [];
$body    = $input['body'] ?? null;

// Validate URL is a known translation endpoint
$allowedHosts = [
    'api.mymemory.translated.net',
    'lingva.ml',
    'libretranslate.com',
    'translate.argosopentech.com',
    'translation.googleapis.com',
    'api-free.deepl.com',
    'api.cognitive.microsofttranslator.com',
    'api.modernmt.com',
];
$urlHost = parse_url($url, PHP_URL_HOST);
if (!in_array($urlHost, $allowedHosts)) {
    http_response_code(400);
    echo json_encode(['error' => 'Host not allowed: ' . $urlHost]);
    exit;
}

// Build curl request
$ch = curl_init($url);
curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_TIMEOUT        => 15,
    CURLOPT_FOLLOWLOCATION => false,
    CURLOPT_SSL_VERIFYPEER => true,
    CURLOPT_USERAGENT      => 'PDFPro-Proxy/1.0',
]);

$curlHeaders = [];
foreach ($headers as $k => $v) {
    $curlHeaders[] = "$k: $v";
}

if ($method === 'POST') {
    curl_setopt($ch, CURLOPT_POST, true);
    if ($body) {
        if (is_array($body)) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($body));
        } else {
            curl_setopt($ch, CURLOPT_POSTFIELDS, is_string($body) ? $body : json_encode($body));
        }
    }
} elseif ($method === 'GET') {
    curl_setopt($ch, CURLOPT_HTTPGET, true);
}

if ($curlHeaders) {
    curl_setopt($ch, CURLOPT_HTTPHEADER, $curlHeaders);
}

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$error    = curl_error($ch);
curl_close($ch);

if ($error) {
    http_response_code(502);
    echo json_encode(['error' => 'Proxy error: ' . $error]);
    exit;
}

http_response_code($httpCode);
echo $response;
